(() => {
  const doc = document;

  const ready = (fn) => {
    if (doc.readyState !== 'loading') {
      fn();
    } else {
      doc.addEventListener('DOMContentLoaded', fn, { once: true });
    }
  };

  const clamp = (value, min, max) => Math.min(Math.max(value, min), max);

  const animateValue = (el, target) => {
    const duration = 1200;
    const startTime = performance.now();
    const start = 0;

    const frame = (now) => {
      const progress = clamp((now - startTime) / duration, 0, 1);
      const eased = 1 - Math.pow(1 - progress, 3);
      const current = Math.floor(start + (target - start) * eased);
      el.textContent = target >= 1000 ? current.toLocaleString() : current;
      if (progress < 1) {
        requestAnimationFrame(frame);
      } else {
        el.textContent = target >= 1000 ? target.toLocaleString() : target;
      }
    };

    requestAnimationFrame(frame);
  };

  const setupReveal = () => {
    const revealables = doc.querySelectorAll('[data-animate]');
    if (!revealables.length) return;

    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            entry.target.classList.add('is-visible');
            observer.unobserve(entry.target);
          }
        });
      },
      { threshold: 0.18, rootMargin: '0px 0px -12%' }
    );

    revealables.forEach((el) => observer.observe(el));
  };

  const setupTilt = () => {
    const tiltables = doc.querySelectorAll('[data-tilt]');
    if (!tiltables.length) return;

    tiltables.forEach((el) => {
      el.addEventListener('pointermove', (event) => {
        const rect = el.getBoundingClientRect();
        const x = (event.clientX - rect.left) / rect.width;
        const y = (event.clientY - rect.top) / rect.height;
        const rotateY = clamp((x - 0.5) * 16, -10, 10);
        const rotateX = clamp((0.5 - y) * 16, -10, 10);
        el.style.setProperty('--tilt-x', `${rotateY}deg`);
        el.style.setProperty('--tilt-y', `${rotateX}deg`);
      });

      el.addEventListener('pointerleave', () => {
        el.style.removeProperty('--tilt-x');
        el.style.removeProperty('--tilt-y');
      });
    });
  };

  const setupCounters = () => {
    const counters = doc.querySelectorAll('[data-stat-target]');
    if (!counters.length) return;

    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          if (entry.isIntersecting) {
            const target = Number(entry.target.dataset.statTarget || '0');
            animateValue(entry.target, target);
            observer.unobserve(entry.target);
          }
        });
      },
      { threshold: 0.6 }
    );

    counters.forEach((el) => observer.observe(el));
  };

  const setupSlider = () => {
    const slider = doc.querySelector('[data-slider]');
    if (!slider) return;

    const track = slider.querySelector('.feature-slider__track');
    const slides = slider.querySelectorAll('.feature-slide');
    if (slides.length <= 1) {
      slider.querySelector('.feature-slider__controls')?.classList.add('is-hidden');
      slider.querySelector('.feature-slider__dots')?.classList.add('is-hidden');
      return;
    }

    const dots = slider.querySelectorAll('[data-slider-dot]');
    const prevBtn = slider.querySelector('[data-slider-prev]');
    const nextBtn = slider.querySelector('[data-slider-next]');

    let index = 0;
    let timer = null;

    const activate = (idx) => {
      index = (idx + slides.length) % slides.length;
      track.style.transform = `translateX(-${index * 100}%)`;
      dots.forEach((dot, dotIndex) => {
        dot.classList.toggle('is-active', dotIndex === index);
      });
    };

    const next = () => {
      activate(index + 1);
    };

    const previous = () => {
      activate(index - 1);
    };

    const reset = () => {
      if (timer) clearInterval(timer);
      timer = setInterval(next, 6000);
    };

    dots.forEach((dot, dotIndex) => {
      dot.addEventListener('click', () => {
        activate(dotIndex);
        reset();
      });
    });

    prevBtn?.addEventListener('click', () => {
      previous();
      reset();
    });

    nextBtn?.addEventListener('click', () => {
      next();
      reset();
    });

    slider.addEventListener('pointerenter', () => timer && clearInterval(timer));
    slider.addEventListener('pointerleave', () => reset());

    activate(0);
    reset();
  };

  const setupScrollSpy = () => {
    const links = Array.from(doc.querySelectorAll('[data-scroll]'));
    const sections = links
      .map((link) => {
        const target = doc.querySelector(link.getAttribute('href'));
        return target && target.tagName === 'SECTION' ? target : null;
      })
      .filter(Boolean);

    if (!links.length || !sections.length) return;

    links.forEach((link) =>
      link.addEventListener('click', (event) => {
        event.preventDefault();
        const target = doc.querySelector(link.getAttribute('href'));
        if (target) {
          target.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
      })
    );

    const observer = new IntersectionObserver(
      (entries) => {
        const visible = entries
          .filter((entry) => entry.isIntersecting)
          .sort((a, b) => b.intersectionRatio - a.intersectionRatio)[0];

        if (!visible) return;

        const activeId = `#${visible.target.id}`;
        links.forEach((link) => link.classList.toggle('is-active', link.getAttribute('href') === activeId));
      },
      { threshold: 0.4, rootMargin: '-20% 0px -40% 0px' }
    );

    sections.forEach((section) => observer.observe(section));
  };

  const setupPageModal = () => {
    const modal = doc.querySelector('[data-page-modal]');
    if (!modal) return;

    const cards = doc.querySelectorAll('[data-page-card]');
    const title = modal.querySelector('[data-page-modal-title]');
    const summary = modal.querySelector('[data-page-modal-summary]');
    const link = modal.querySelector('[data-page-modal-link]');
    const close = modal.querySelector('[data-modal-close]');

    const open = (card) => {
      title.textContent = card.dataset.pageTitle || '';
      summary.textContent = card.dataset.pageSummary || '';
      const href = card.dataset.pageLink || '#';
      if (href === '#') {
        link.classList.add('is-hidden');
      } else {
        link.classList.remove('is-hidden');
        link.setAttribute('href', href);
      }
      if (typeof modal.showModal === 'function') {
        modal.showModal();
      }
    };

    const closeModal = () => {
      if (typeof modal.close === 'function') {
        modal.close();
      }
    };

    cards.forEach((card) => {
      card.addEventListener('click', () => open(card));
      card.addEventListener('keydown', (event) => {
        if (event.key === 'Enter' || event.key === ' ') {
          event.preventDefault();
          open(card);
        }
      });
    });

    close?.addEventListener('click', closeModal);
    modal.addEventListener('click', (event) => {
      if (event.target === modal) closeModal();
    });
  };

  ready(() => {
    setupReveal();
    setupTilt();
    setupCounters();
    setupSlider();
    setupScrollSpy();
    setupPageModal();
  });
})();
