const root = document.documentElement;

const clamp = (value, min, max) => Math.min(Math.max(value, min), max);

const ready = (callback) => {
  if (document.readyState !== 'loading') {
    callback();
  } else {
    document.addEventListener('DOMContentLoaded', callback, { once: true });
  }
};

const animateCounter = (el) => {
  const target = Number(el.dataset.counterTarget) || 0;
  const duration = 1600;
  const startTime = performance.now();

  const tick = (now) => {
    const progress = clamp((now - startTime) / duration, 0, 1);
    const eased = 1 - Math.pow(1 - progress, 3);
    const value = Math.floor(target * eased);
    el.textContent = target >= 1000 ? `${value.toLocaleString()}${el.dataset.counterSuffix || ''}` : `${value}${el.dataset.counterSuffix || ''}`;
    if (progress < 1) {
      requestAnimationFrame(tick);
    } else {
      el.textContent = target >= 1000 ? `${target.toLocaleString()}${el.dataset.counterSuffix || ''}` : `${target}${el.dataset.counterSuffix || ''}`;
    }
  };

  requestAnimationFrame(tick);
};

const setupCounters = () => {
  const counters = document.querySelectorAll('[data-counter]');
  if (!counters.length) return;

  const observer = new IntersectionObserver(
    (entries, obs) => {
      entries.forEach((entry) => {
        if (!entry.isIntersecting) return;
        animateCounter(entry.target);
        obs.unobserve(entry.target);
      });
    },
    { threshold: 0.6 }
  );

  counters.forEach((counter) => observer.observe(counter));
};

const setupReveal = () => {
  const revealables = document.querySelectorAll('[data-animate]');
  if (!revealables.length) return;

  const observer = new IntersectionObserver(
    (entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) entry.target.classList.add('is-visible');
      });
    },
    { threshold: 0.18, rootMargin: '0px 0px -12%' }
  );

  revealables.forEach((el) => observer.observe(el));
};

const setupTilt = () => {
  const tiltElements = document.querySelectorAll('[data-tilt]');
  if (!tiltElements.length) return;

  tiltElements.forEach((el) => {
    el.addEventListener('pointermove', (event) => {
      const rect = el.getBoundingClientRect();
      const x = (event.clientX - rect.left) / rect.width;
      const y = (event.clientY - rect.top) / rect.height;
      const rotateY = clamp((x - 0.5) * 16, -12, 12);
      const rotateX = clamp((0.5 - y) * 16, -12, 12);
      el.style.transform = `perspective(900px) rotateX(${rotateX}deg) rotateY(${rotateY}deg) translateZ(12px)`;
    });

    el.addEventListener('pointerleave', () => {
      el.style.transform = '';
    });
  });
};

const setupPricingToggle = () => {
  const pills = document.querySelectorAll('.pricing__pill');
  const glow = document.querySelector('.pricing__glow');
  const tiers = document.querySelectorAll('.tier');
  if (!pills.length || !glow || !tiers.length) return;

  const activate = (billing) => {
    pills.forEach((pill, index) => {
      const isActive = pill.dataset.billing === billing;
      pill.classList.toggle('is-active', isActive);
      if (isActive) glow.style.transform = `translateX(${index * 100}%)`;
    });

    tiers.forEach((tier) => {
      const monthly = tier.querySelector('[data-price-monthly]');
      const annual = tier.querySelector('[data-price-annual]');
      if (!monthly || !annual) return;
      if (billing === 'monthly') {
        monthly.style.display = 'inline';
        annual.style.display = 'none';
      } else {
        monthly.style.display = 'none';
        annual.style.display = 'inline';
      }
    });
  };

  pills.forEach((pill) => {
    pill.addEventListener('click', () => activate(pill.dataset.billing));
  });

  activate('monthly');
};

const setupCarousel = () => {
  const track = document.querySelector('[data-carousel]');
  if (!track) return;

  let offset = 0;
  const slides = track.children;
  const total = slides.length;
  if (total <= 1) return;

  setInterval(() => {
    offset = (offset + 1) % total;
    track.style.transform = `translateX(calc(-${offset} * (100% + 24px)))`;
  }, 5500);
};

const setupParallax = () => {
  const scene = document.querySelector('[data-parallax]');
  if (!scene) return;

  window.addEventListener('scroll', () => {
    const rect = scene.getBoundingClientRect();
    const progress = clamp(1 - rect.top / window.innerHeight, 0, 1);
    scene.style.transform = `translateY(${progress * -24}px) scale(${1 + progress * 0.05})`;
  });
};

ready(() => {
  setupReveal();
  setupCounters();
  setupTilt();
  setupPricingToggle();
  setupCarousel();
  setupParallax();
});
